// https://github.com/mapbox/node-sqlite3/wiki/API
// https://www.techiediaries.com/node-sqlite-crud/
const sqlite3 = require( 'sqlite3' ).verbose();
const app_config = require( './../configs/app_config' );
const easyUid = require( './../method/easyUid.js' );
const db = new sqlite3.Database( `${ app_config.record_path() }/sr1.db` );
const cbLog = ( err ) => err ? console.log( err ) : null;
db.serialize( () => {
  db.run( 'create table if not exists params (name TEXT, value TEXT)', cbLog );
  db.run( 'CREATE UNIQUE INDEX IF NOT EXISTS host_name ON params (name)', cbLog );
  db.run( `insert into params (name , value ) values ('host_uid', '${ easyUid() }') on conflict do nothing`, cbLog );
  const sql_table = `CREATE TABLE IF NOT EXISTS records
                     (
                         record_created TEXT,
                         file           TEXT,
                         naumen_id      TEXT,
                         session_id     TEXT,
                         is_sent        INTEGER DEFAULT 0,
                         is_deleted     INTEGER DEFAULT 0,
                         login          TEXT
                     )`;
  db.run( sql_table, cbLog );
  db.run( `CREATE INDEX IF NOT EXISTS index_is_sent ON records (is_sent)`, cbLog );
  db.run( `CREATE INDEX IF NOT EXISTS index_file ON records (file)`, cbLog );
  // migrate 2
  db.run( `DROP INDEX IF EXISTS records_naumen_id_session_id`, cbLog );
  db.run( `CREATE UNIQUE INDEX IF NOT EXISTS records_record_created_session_id ON records (record_created, session_id)`, cbLog );
  let stmt = db.prepare( `pragma table_info(records)` );
  stmt.all( ( err, rows ) => {
    if ( err ) return err;
    const fields = rows.map( row => row.name );
    if ( !fields.includes( 'record_created' ) ) {
      db.run( 'ALTER TABLE records ADD COLUMN record_created TEXT;' );
      console.log( 'add field record_created' )
    }
    // migrate 3
    if ( !fields.includes( 'login' ) ) {
      db.run( 'ALTER TABLE records ADD COLUMN login TEXT;' );
      console.log( 'add field login' )
    }
    // migrate 4
    if ( !fields.includes( 'cached' ) ) {
      db.run( 'ALTER TABLE records ADD COLUMN cached INTEGER DEFAULT 0;' );
      console.log( 'add field cached' )
    }
  } );
} );

module.exports = {
  getRecordsBySessionIds( { ids } ) {
    return new Promise( ( resolve, reject ) => {
      if ( ids.length === 0 ) return reject( null );
      const ids_where = '\'' + ids.join( '\',\'' ) + '\'';
      const query = `select * from records as r where session_id in (${ ids_where })`;
      db.all( query, ( err, rows ) => {
        if ( err ) {
          return reject( err );
        }
        if ( rows.length > 0 ) {
          return resolve( rows );
        }
        reject( null );
      } );
    } );
  },
  cachedRecord( item ) {
    console.log( 'cachedRecord', item.file );
    const query = `update records
                   set cached = 1
                   where session_id = ?
                     and record_created = ?`;
    let stmt = db.prepare( query );
    stmt.run( item.session_id, item.record_created );
    stmt.finalize();
  },
  getNotCachedRecords() {
    const query = `select file, session_id, record_created
                   from records as r
                   where r.is_deleted = 0
                     and r.cached = 0
                   order by session_id`;
    return new Promise( ( resolve, reject ) => {
      db.all( query, ( err, rows ) => {
        if ( err ) {
          reject( err );
          return false;
        }
        resolve( rows );
      } );
    } );
  },
  get_nosent_records() {
    const query = `select *
                   from records as r
                   where r.is_sent = 0
                   order by r.record_created desc
                   limit 200`;
    return new Promise( ( resolve, reject ) => {
      db.all( query, ( err, rows ) => {
        if ( err ) {
          reject( err );
          return false;
        }
        resolve( rows );
      } );
    } );
  },
  get_not_deleted_records() {
    const query = `select file, session_id, naumen_id, record_created
                   from records as r
                   where r.is_deleted = 0
                     and r.is_sent = 1
                   order by session_id`;
    return new Promise( ( resolve, reject ) => {
      db.all( query, ( err, rows ) => {
        if ( err ) {
          reject( err );
          return false;
        }
        resolve( rows );
      } );
    } );
  },
  complete_sent_record( { session_id, record_created } ) {
    const query = `update records
                   set is_sent = 1
                   where session_id = ?
                     and record_created = ?`;
    let stmt = db.prepare( query );
    stmt.run( session_id, record_created );
    stmt.finalize();
  },
  set_delete_record( { record_created, session_id } ) {
    const query = `update records
                   set is_deleted = 1
                   where record_created = ?
                     and session_id = ?`;
    let stmt = db.prepare( query );
    stmt.run( record_created, session_id );
    stmt.finalize();
  },
  addRecord,
  getHostUid
};

function addRecord( { record_created, file, naumen_id, session_id, login } ) {
  const query = `insert into records ( record_created, file, naumen_id, session_id, login )
                 values ( ?, ?, ?, ?, ? )
                 on conflict (record_created, session_id) do update
                     set is_deleted = 0`;
  let stmt = db.prepare( query );
  stmt.run( record_created, file, naumen_id, session_id, login );
  stmt.finalize();
}

function getHostUid() {
  const query = `select p.value
                 from params as p
                 where p.name = 'host_uid'`;
  return new Promise( ( resolve, reject ) => {
    db.get( query, function( err, row ) {
      if ( err ) return reject( err );
      if ( row ) return resolve( row.value );
      resolve( null );
    } );
  } );
}